package gov.va.med.mhv.sm.admin.service.impl;

import java.util.List;

import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Consumes;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Service;
import org.springframework.beans.factory.annotation.Autowired;

import gov.va.med.mhv.sm.admin.dto.PcmmProviderDTO;
import gov.va.med.mhv.sm.admin.data.enums.SMErrorEnum;
import gov.va.med.mhv.sm.admin.exception.SMApiException;
import gov.va.med.mhv.sm.admin.jpa.model.TriageRelation;
import gov.va.med.mhv.sm.admin.service.util.ResponseCodeUtil;
import gov.va.med.mhv.sm.admin.converter.TriageRelationConverter;
import gov.va.med.mhv.sm.admin.jpa.repository.TriageRelationRepository;
import gov.va.med.mhv.sm.admin.service.util.SMHealthshareDelegate;
import gov.va.med.mhv.sm.healthshare.wsclient.adminqueries.Provider;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.ArraySchema;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.responses.ApiResponse;


@Path("/pcmmproviders")
@Service("pcmmproviderService")
@Tag(name = "PCMM Provider Service")
@Consumes(MediaType.APPLICATION_JSON)
@Produces(MediaType.APPLICATION_JSON)
@Component
public class PCMMProviderService { 

    @Autowired
    private TriageRelationRepository triageRelationRepository;

    @Autowired
    private SMHealthshareDelegate smHealthShareDelegate;
    
    private static final Log log = LogFactory.getLog(PCMMProviderService.class);

	@GET
	@Operation(summary = "Get PCMM Providers based on Query Parameters provided. ")
	@ApiResponses(value = {
	        @ApiResponse(responseCode = "200",
	        			content = @Content(mediaType = MediaType.APPLICATION_JSON,array = @ArraySchema(schema = @Schema(implementation = PcmmProviderDTO.class)))),
			@ApiResponse(responseCode = "404", description = "PCMM Providers are not found",
						content = @Content(mediaType = MediaType.APPLICATION_JSON, schema = @Schema(implementation = SMApiException.class))),
			@ApiResponse(responseCode = "500", description = "Unexpected Error Occurred",
						content = @Content(mediaType = MediaType.APPLICATION_JSON, schema = @Schema(implementation = SMApiException.class)))
	})
	public List<PcmmProviderDTO> getPCMMProviders(
												@QueryParam("triageGroupId") Long triageGroupId, 
												@QueryParam("lastName") String lastName,
												@QueryParam("firstName") String firstName, 
												@QueryParam("station") String station)  throws SMApiException {
		try{
			PCMMProviderEnum.SearchEnum pcmmQueryEnum = validateQueryParams(triageGroupId,station,lastName,firstName);
			switch (pcmmQueryEnum) {
					case GET_PCMM_PROVIDER_BY_TRIAGE_GROUP_ID:
						return getPCMMProvidersByTriageGroup(triageGroupId);
					case GET_PCMM_PROVIDER_BY_STATION_AND_NAME:
						return getPCMMProvidersFromHSByNameAndStation(lastName, firstName, station);
					default:
						throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);		
			}
		}catch(Exception pcmmExp){
			log.error(pcmmExp);
			throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR,pcmmExp);
		}
		// doValidate Query Params - throws an exception if query params are not valid
		// identify case (by triage group or by station, first, last
		// if(byTriageGroupId) { call getPCMMProvidersByTriageGroup method; }
		// else if (byStationLastFirst) { call getPCMMProvidersFromHSByNameAndStation method; }
	}
	
	private PCMMProviderEnum.SearchEnum validateQueryParams(Long triageGroupId, String station, String lastName, String firstName) throws SMApiException{
		if(triageGroupId !=null){
			if(station !=null || lastName !=null || firstName !=null){
				throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
			}else{
				return PCMMProviderEnum.SearchEnum.GET_PCMM_PROVIDER_BY_TRIAGE_GROUP_ID;
			}
		}else if(station !=null){
			if(triageGroupId !=null){
				throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
			}else{
				if(lastName == null){
					throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
				}else{
					return PCMMProviderEnum.SearchEnum.GET_PCMM_PROVIDER_BY_STATION_AND_NAME;
				}
			}
		}
		else{
			throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
		}
	}
	
	private List<PcmmProviderDTO> getPCMMProvidersByTriageGroup(Long triageGroupId)  throws SMApiException {
		
		List<TriageRelation> pcmmProviderList = null;
		List<PcmmProviderDTO> pcmmProviderDTOList = null;
		try {	
			if(triageGroupId != null) {
				pcmmProviderList = triageRelationRepository.getPCMMProvidersByTriageGroup(triageGroupId);
				
			if(pcmmProviderList != null || pcmmProviderList.size() >0) {
				pcmmProviderDTOList = TriageRelationConverter.convertPCMMProvidersList(pcmmProviderList);
			}else {
				throw new SMApiException(SMErrorEnum.PCMM_PROVIDERS_NOT_FOUND.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
			}
		}
		} catch (Exception pcmmExp) {
			log.error(pcmmExp);
			throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR,pcmmExp);
		}

		return pcmmProviderDTOList;
	}
	
	private List<PcmmProviderDTO> getPCMMProvidersFromHSByNameAndStation(
																		String lastName,
																		String firstName, 
																		String facilityId) throws SMApiException {
		List<Provider> pcmmProviderList = null;
		List<PcmmProviderDTO> pcmmProviderDTOList = null;
		try{	
			pcmmProviderList = smHealthShareDelegate.getPCMMProvidersByName(lastName,firstName,facilityId);
			if(pcmmProviderList != null && pcmmProviderList.size()>0) {
				pcmmProviderDTOList = TriageRelationConverter.convertHSPCMMProvidersList(pcmmProviderList);
			}else {
				throw new SMApiException(SMErrorEnum.PCMM_PROVIDERS_NOT_FOUND.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
			}
		} catch (Exception pcmmExp) {
			if(log.isErrorEnabled()){
				log.error(pcmmExp);
				throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR,pcmmExp);
			}
		}
		return pcmmProviderDTOList;
	}
	
	private static class PCMMProviderEnum{
		public enum SearchEnum{
			GET_PCMM_PROVIDER_BY_TRIAGE_GROUP_ID,
			GET_PCMM_PROVIDER_BY_STATION_AND_NAME;
		}
	}
	
}
